import { useState, useEffect } from 'react'
import { BookOpen, Calendar, Award, RefreshCw, Users } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyOrders.css'

const sliderImages = [
  '/images/library-1.jpg',
  '/images/library-2.jpg',
  '/images/library-3.jpg',
  '/images/library-4.jpg'
]

function MyOrders() {
  const [orders, setOrders] = useState({
    active: [],
    past: []
  })

  useEffect(() => {
    const saved = localStorage.getItem('libraryOrders')
    if (saved) {
      setOrders(JSON.parse(saved))
    } else {
      const defaultOrders = {
        active: [
          {
            id: 1,
            title: 'Война и мир',
            genre: 'Художественная',
            returnDate: '2025-03-25',
            status: 'active'
          },
          {
            id: 2,
            title: 'Физика для начинающих',
            genre: 'Научная',
            returnDate: '2025-03-28',
            status: 'active'
          },
          {
            id: 3,
            title: 'История России',
            genre: 'Учебник',
            returnDate: '2025-03-30',
            status: 'active'
          }
        ],
        past: [
          {
            id: 4,
            title: 'Математика для детей',
            genre: 'Детская',
            returnDate: '2025-03-15',
            status: 'completed'
          },
          {
            id: 5,
            title: 'Основы программирования',
            genre: 'Научная',
            returnDate: '2025-03-10',
            status: 'completed'
          },
          {
            id: 6,
            title: 'Приключения Тома Сойера',
            genre: 'Детская',
            returnDate: '2025-03-05',
            status: 'completed'
          }
        ]
      }
      setOrders(defaultOrders)
      localStorage.setItem('libraryOrders', JSON.stringify(defaultOrders))
    }

    const interval = setInterval(() => {
      const updated = localStorage.getItem('libraryOrders')
      if (updated) {
        setOrders(JSON.parse(updated))
      }
    }, 500)

    return () => clearInterval(interval)
  }, [])

  const extendOrder = (orderId) => {
    const updated = { ...orders }
    const order = updated.active.find(o => o.id === orderId)
    if (order) {
      const currentDate = new Date(order.returnDate)
      currentDate.setDate(currentDate.getDate() + 7)
      order.returnDate = currentDate.toISOString().split('T')[0]
      setOrders(updated)
      localStorage.setItem('libraryOrders', JSON.stringify(updated))
    }
  }

  const formatDate = (dateString) => {
    const date = new Date(dateString)
    return date.toLocaleDateString('ru-RU', { day: 'numeric', month: 'long', year: 'numeric' })
  }

  return (
    <div className="my-orders">
      <div className="page-header">
        <div className="header-shapes">
          <div className="shape shape-1"></div>
          <div className="shape shape-2"></div>
          <div className="shape shape-3"></div>
          <div className="shape shape-4"></div>
        </div>
        <h1>Мои заказы</h1>
        <p>Управляйте своими забронированными книгами</p>
      </div>

      <ImageSlider images={sliderImages} />

      <div className="stats-section">
        <div className="stats-grid">
          <div className="stat-card">
            <BookOpen className="stat-icon" size={32} />
            <div className="stat-content">
              <div className="stat-value">{orders.active.length}</div>
              <div className="stat-label">Активных заказов</div>
            </div>
          </div>
          <div className="stat-card">
            <Calendar className="stat-icon" size={32} />
            <div className="stat-content">
              <div className="stat-value">{orders.past.length}</div>
              <div className="stat-label">Прочитано книг</div>
            </div>
          </div>
          <div className="stat-card">
            <Award className="stat-icon" size={32} />
            <div className="stat-content">
              <div className="stat-value">14</div>
              <div className="stat-label">Дней на возврат</div>
            </div>
          </div>
        </div>
      </div>

      <div className="orders-section">
        <h2 className="section-title">Активные заказы</h2>
        {orders.active.length === 0 ? (
          <div className="empty-state">
            <BookOpen className="empty-icon" size={64} />
            <p>У вас нет активных заказов</p>
          </div>
        ) : (
          <div className="orders-grid">
            {orders.active.map(order => (
              <div key={order.id} className="order-card">
                <div className="order-header">
                  <h3>{order.title}</h3>
                  <span className="genre-badge">{order.genre}</span>
                </div>
                <div className="order-details">
                  <div className="detail-item">
                    <Calendar className="detail-icon" size={20} />
                    <span>Дата возврата: {formatDate(order.returnDate)}</span>
                  </div>
                </div>
                <button 
                  className="extend-btn"
                  onClick={() => extendOrder(order.id)}
                >
                  <RefreshCw className="extend-icon" size={20} />
                  <span>Продлить на 7 дней</span>
                </button>
              </div>
            ))}
          </div>
        )}
      </div>

      <div className="info-section">
        <h2 className="section-title">Почему выбирают нас?</h2>
        <div className="info-grid">
          <div className="info-card">
            <BookOpen className="info-icon" size={40} />
            <h3>Большая коллекция</h3>
            <p>Более 500 000 книг различных жанров</p>
          </div>
          <div className="info-card">
            <Users className="info-icon" size={40} />
            <h3>Опытные библиотекари</h3>
            <p>Профессиональная помощь в подборе литературы</p>
          </div>
          <div className="info-card">
            <Award className="info-icon" size={40} />
            <h3>Удобный сервис</h3>
            <p>Онлайн-бронирование и продление книг</p>
          </div>
        </div>
      </div>
    </div>
  )
}

export default MyOrders

